// Use `ext` as a cross-browser alias for `chrome` or `browser`.
const ext = globalThis.browser ?? chrome;

(function() {
    'use strict';

    // Avoid running in iframes
    if (window.self !== window.top) {
        return;
    }

    /**
     * This function contains all the profile selection UI logic.
     * It will only be executed if the script is running in a tab
     * managed by the extension's popup window.
     */
    function initializeProfileSelector() {
        // --- START MOBILE DETECTION ---
        // Fungsi deteksi seluler telah dihapus untuk mendukung browser seluler.
        // --- END MOBILE DETECTION ---

        // If profile is already selected in this session, do nothing.
        if (sessionStorage.getItem('chatgpt_profile_selected')) {
            return;
        }

        const PROFILE_URLS = {
            '1': 'https://instanpro.com/kukis/1toolgram/1.json',
            '2': 'https://instanpro.com/kukis/1toolgram/2.json',
        };

        function showModal() {
            const overlay = document.createElement('div');
            overlay.id = 'profile-selector-overlay';
            
            overlay.innerHTML = `
                <div id="profile-selector-modal">
                    <h2>Pilih Profil</h2>
                    <p>Silakan pilih profil untuk melanjutkan ke ChatGPT.</p>
                    <div class="profile-selector-buttons">
                        <button data-profile="1">Profil 1</button>
                        <button data-profile="2">Profil 2</button>
                    </div>
                    <div id="profile-selector-status"></div>
                    <div class="profile-selector-loader"></div>
                </div>
            `;

            const injectModal = () => {
                if (document.body) {
                    document.body.style.overflow = 'hidden';
                    document.body.appendChild(overlay);
                }
                document.documentElement.style.visibility = 'visible';
                overlay.querySelector('.profile-selector-buttons').addEventListener('click', handleProfileSelection);
            };

            // Hide page content until modal is ready
            document.documentElement.style.visibility = 'hidden';

            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', injectModal, { once: true });
            } else {
                injectModal();
            }
        }

        async function handleProfileSelection(event) {
            if (event.target.tagName !== 'BUTTON') return;
            
            const profileId = event.target.dataset.profile;
            if (!profileId) return;

            const statusEl = document.getElementById('profile-selector-status');
            const loader = document.querySelector('.profile-selector-loader');
            const buttons = document.querySelector('.profile-selector-buttons');

            buttons.style.display = 'none';
            loader.style.display = 'block';
            statusEl.textContent = 'Menyiapkan profil Anda... Mohon tunggu.';

            try {
                const response = await ext.runtime.sendMessage({
                    type: 'SET_PROFILE_COOKIES',
                    payload: {
                        profileUrl: PROFILE_URLS[profileId],
                    }
                });

                if (response && response.success) {
                    statusEl.textContent = 'Profil berhasil dimuat. Mengalihkan...';
                    sessionStorage.setItem('chatgpt_profile_selected', 'true');
                    // Give a moment for user to see the message
                    setTimeout(() => {
                        window.location.reload();
                    }, 1500);
                } else {
                    throw new Error(response?.error || 'Gagal memuat profil.');
                }
            } catch (error) {
                statusEl.textContent = `Error: ${error.message}`;
                loader.style.display = 'none';
                statusEl.textContent += " Silakan muat ulang halaman dan coba lagi.";
                if (document.body) {
                  document.body.style.overflow = 'auto';
                }
            }
        }

        showModal();
    }

    /**
     * Checks if the current tab is managed by the extension.
     * Caches the result in sessionStorage to avoid repeated checks.
     * If managed, it calls the function to display the profile selector.
     */
    function checkAndInitialize() {
        const isManagedSession = sessionStorage.getItem('isManagedByGudangbot');

        if (isManagedSession === 'true') {
            initializeProfileSelector();
        } else if (isManagedSession === 'false') {
            // It's not a managed tab, do nothing.
            return;
        } else {
            // Flag not set, ask the background script.
            if (ext && ext.runtime && typeof ext.runtime.sendMessage === 'function') {
                ext.runtime.sendMessage({ type: 'CHECK_IF_MANAGED' }, (response) => {
                    if (ext.runtime.lastError) {
                        console.warn("Gudangbot: ", ext.runtime.lastError.message);
                        sessionStorage.setItem('isManagedByGudangbot', 'false');
                        return;
                    }
                    
                    const isManaged = !!(response && response.isManaged);
                    sessionStorage.setItem('isManagedByGudangbot', String(isManaged));

                    if (isManaged) {
                        initializeProfileSelector();
                    } else {
                        // If not managed, make sure the page is visible, as the modal logic might hide it initially.
                        document.documentElement.style.visibility = 'visible';
                    }
                });
            } else {
                // Assume not managed if extension APIs aren't available.
                sessionStorage.setItem('isManagedByGudangbot', 'false');
                document.documentElement.style.visibility = 'visible';
            }
        }
    }

    checkAndInitialize();
})();