// Use `ext` as a cross-browser alias for `chrome` or `browser`.
const ext = globalThis.browser ?? chrome;

const TARGET_URL = "https://test.teknoaiglobal.com/";

// --- START: AUTOMATIC CLEANUP LOGIC ---
// The following functions handle the automatic clearing of cookies
// when the extension's popup window is closed.

// Function to clear ChatGPT cookies
async function clearChatGPTCookies() {
    const CHATGPT_DOMAIN = "chatgpt.com";
    console.log("Clearing ChatGPT cookies...");
    try {
        // This will get cookies for chatgpt.com and its subdomains (like sora.chatgpt.com)
        const cookies = await ext.cookies.getAll({ domain: CHATGPT_DOMAIN });
        if (cookies.length === 0) {
            console.log("No ChatGPT cookies to clear.");
            return;
        }

        for (const cookie of cookies) {
            const protocol = cookie.secure ? "https://" : "http://";
            // Domain may start with a dot, remove it for the URL
            const domain = cookie.domain.startsWith('.') ? cookie.domain.substring(1) : cookie.domain;
            const cookieUrl = `${protocol}${domain}${cookie.path}`;
            await ext.cookies.remove({ url: cookieUrl, name: cookie.name });
        }
        console.log(`Successfully cleared ${cookies.length} ChatGPT cookies.`);
    } catch (error) {
        console.error("Error clearing ChatGPT cookies:", error);
    }
}

// Function to clear recent browsing history
async function clearRecentHistory() {
    const FORTY_EIGHT_HOURS_IN_MS = 48 * 60 * 60 * 1000;
    const endTime = Date.now();
    const startTime = endTime - FORTY_EIGHT_HOURS_IN_MS;

    console.log("Clearing browsing history from the last 48 hours...");
    try {
        await ext.history.deleteRange({
            startTime: startTime,
            endTime: endTime,
        });
        console.log("Successfully cleared browsing history from the last 48 hours.");
    } catch (error) {
        console.error("Error clearing recent history:", error);
    }
}


/**
 * Main cleanup function that orchestrates clearing cookies and history.
 */
async function performCleanup() {
    console.log("Cleanup process initiated.");
    // Clear ChatGPT Cookies for a clean session next time.
    await clearChatGPTCookies();
    // Clear browsing history from the last 48 hours.
    await clearRecentHistory();
    console.log("Cleanup process completed.");
}

// A helper function to run cleanup and clear the stored window ID to prevent re-triggering.
async function performCleanupAndClearStorage() {
    await performCleanup();
    await ext.storage.local.remove('popupWindowId');
    console.log("Stored window ID cleared after cleanup.");
}

// --- END: AUTOMATIC CLEANUP LOGIC ---

// --- START: EVENT LISTENERS ---

// Listener for tab updates. This now triggers cleanup on every navigation
// within the managed window.
ext.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
    // We only care about top-level frame navigation and when the tab has finished loading.
    if (changeInfo.status !== 'complete' || !tab.url) {
        return;
    }

    const data = await ext.storage.local.get('popupWindowId');
    const storedWindowId = data.popupWindowId;

    // If navigation occurs within the managed window, perform cleanup.
    if (storedWindowId && tab.windowId === storedWindowId) {
        console.log(`Navigation in managed window to: ${tab.url}. Performing cleanup...`);
        await performCleanup();
    }
});


ext.action.onClicked.addListener(async () => {
    // Use `ext.storage.local` for cross-browser compatibility.
    const data = await ext.storage.local.get('popupWindowId');
    let popupWindowId = data.popupWindowId;

    if (popupWindowId) {
        try {
            // Check if window still exists
            await ext.windows.get(popupWindowId);
            await ext.windows.update(popupWindowId, { focused: true });
            return;
        } catch (error) {
            // Window doesn't exist, so we'll create a new one.
            popupWindowId = null;
        }
    }

    // If no window exists, create a new one.
    console.log("Creating new extension window.");

    try {
        const window = await ext.windows.create({
            url: TARGET_URL,
            type: "popup",
            width: 600,
            height: 1200,
            focused: true
        });
        // Store the ID in local storage
        await ext.storage.local.set({ popupWindowId: window.id });
    } catch (error) {
        console.warn("Failed to create popup window, falling back to tab creation:", error);
        // Fallback for mobile browsers that might not support windows.create type=popup well
        try {
            const tab = await ext.tabs.create({ url: TARGET_URL, active: true });
            // We store the windowId of the tab, so the management logic still works (mostly)
            await ext.storage.local.set({ popupWindowId: tab.windowId });
        } catch (tabError) {
            console.error("Failed to create tab as fallback:", tabError);
        }
    }
});

/**
 * Primary listener for cleanup.
 * When the extension's popup window is closed, this function is triggered.
 * It calls the main cleanup routine.
 */
ext.windows.onRemoved.addListener(async (windowId) => {
    const data = await ext.storage.local.get('popupWindowId');
    const storedWindowId = data.popupWindowId;

    if (storedWindowId && windowId === storedWindowId) {
        // The main popup was closed, so perform cleanup automatically.
        console.log("Main extension window closed. Performing automatic cleanup...");
        await performCleanupAndClearStorage();
    }
});

// --- END: EVENT LISTENERS ---


// --- Profile Selector Functionality ---

const delay = (ms) => new Promise(resolve => setTimeout(resolve, ms));
const randomDelay = () => delay(Math.floor(Math.random() * (6000 - 3000 + 1)) + 3000);

async function setProfileCookies(profileUrl, sendResponse) {
    const CHATGPT_DOMAIN = "chatgpt.com";

    try {
        const existingCookies = await ext.cookies.getAll({ domain: CHATGPT_DOMAIN });
        for (const cookie of existingCookies) {
            const protocol = cookie.secure ? "https://" : "http://";
            const domain = cookie.domain.startsWith('.') ? cookie.domain.substring(1) : cookie.domain;
            const cookieUrl = `${protocol}${domain}${cookie.path}`;
            await ext.cookies.remove({ url: cookieUrl, name: cookie.name });
        }

        await randomDelay();

        const cookieResponse = await fetch(profileUrl);
        if (!cookieResponse.ok) {
            throw new Error(`Gagal mengambil data cookies: ${cookieResponse.statusText}`);
        }
        const cookiesToSet = await cookieResponse.json();

        for (const cookie of cookiesToSet) {
            if (!cookie.domain || !cookie.domain.includes(CHATGPT_DOMAIN)) {
                continue;
            }

            const protocol = cookie.secure ? "https://" : "http://";
            const domain = cookie.domain.startsWith('.') ? cookie.domain.substring(1) : cookie.domain;

            const newCookieDetails = {
                url: `${protocol}${domain}${cookie.path || '/'}`,
                name: cookie.name,
                value: cookie.value,
                path: cookie.path,
                secure: cookie.secure,
                httpOnly: cookie.httpOnly,
                sameSite: cookie.sameSite,
                domain: cookie.domain,
            };

            if (cookie.expirationDate) {
                newCookieDetails.expirationDate = cookie.expirationDate;
            }
            if (cookie.storeId) {
                newCookieDetails.storeId = cookie.storeId;
            }

            try {
                await ext.cookies.set(newCookieDetails);
            } catch (e) {
                console.warn(`Tidak dapat mengatur cookie "${cookie.name}":`, e.message);
            }
        }

        await randomDelay();

        sendResponse({ success: true });

    } catch (error) {
        console.error("Error in setProfileCookies:", error);
        sendResponse({ success: false, error: error.message });
    }
}

ext.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'SET_PROFILE_COOKIES') {
        setProfileCookies(message.payload.profileUrl, sendResponse);
        return true;
    }
    if (message.type === 'INJECT_FLOW_COOKIES') {
        (async () => {
            // Use the provided server URL or fallback to default (server 1)
            const DEFAULT_URL = "https://firestore.googleapis.com/v1/projects/tekno-335f8/databases/(default)/documents/artifacts/default-app-id/public/data/public_files/flow-1?key=AIzaSyCirtabCZOy3XMnNLUc-iKIYGegZJbPqhw";
            const FIRESTORE_URL = message.payload?.serverUrl || DEFAULT_URL;
            try {
                const existing = await ext.cookies.getAll({ domain: "labs.google" });
                for (const cookie of existing) {
                    const protocol = cookie.secure ? "https://" : "http://";
                    const domain = cookie.domain.startsWith('.') ? cookie.domain.substring(1) : cookie.domain;
                    const cookieUrl = `${protocol}${domain}${cookie.path}`;
                    await ext.cookies.remove({ url: cookieUrl, name: cookie.name });
                }
                const resp = await fetch(FIRESTORE_URL);
                if (!resp.ok) {
                    throw new Error(`Gagal mengambil data cookies: ${resp.statusText}`);
                }
                const firestoreData = await resp.json();
                const contentStr = firestoreData?.fields?.content?.stringValue;
                if (!contentStr) {
                    throw new Error("Respon Firestore tidak valid");
                }
                const parsed = JSON.parse(contentStr);
                const cookiesToSet = Array.isArray(parsed?.cookies) ? parsed.cookies : [];
                for (const cookie of cookiesToSet) {
                    if (!cookie.domain || !cookie.domain.includes("labs.google")) {
                        continue;
                    }
                    const protocol = cookie.secure ? "https://" : "http://";
                    const domain = cookie.domain.startsWith('.') ? cookie.domain.substring(1) : cookie.domain;
                    const details = {
                        url: `${protocol}${domain}${cookie.path || '/'}`,
                        name: cookie.name,
                        value: cookie.value,
                        path: cookie.path,
                        secure: cookie.secure,
                        httpOnly: cookie.httpOnly,
                        sameSite: cookie.sameSite,
                        domain: cookie.domain,
                    };
                    if (cookie.expirationDate) {
                        details.expirationDate = cookie.expirationDate;
                    }
                    if (cookie.storeId) {
                        details.storeId = cookie.storeId;
                    }
                    try {
                        await ext.cookies.set(details);
                    } catch (e) { }
                }
                sendResponse({ success: true });
            } catch (error) {
                sendResponse({ success: false, error: error.message });
            }
        })();
        return true;
    }

    if (message.type === 'CHECK_IF_MANAGED') {
        (async () => {
            const data = await ext.storage.local.get('popupWindowId');
            const storedWindowId = data.popupWindowId;
            const isManaged = !!(storedWindowId && sender.tab && sender.tab.windowId === storedWindowId);
            sendResponse({ isManaged: isManaged });
        })();
        return true; // Indicate async response
    }
});
